#1. Load Required Packages
required_packages <- c("readxl", "ggplot2", "gridExtra")
for (package in required_packages) {
  if (!requireNamespace(package, quietly = TRUE)) {
    install.packages(package)
  }
  library(package, character.only = TRUE)
}
#Description: This section ensures the required packages (readxl, ggplot2, gridExtra) are installed and loaded.


#2. Import Data Frame
if (!exists("df")) {
  file_path <- "~/COHORT_Prediction_of_CIP.xlsx"
  df <- read_excel(file_path, sheet = "cohort", na = "#N/A")
}
#Description: This section imports data from an Excel file if the data frame df does not already exist.


#3. Filter Data
df_filtered <- df[, c("r_upper_suv_mean","r_upper_suv_sd", 
                      "r_lower_suv_mean","r_lower_suv_sd",
                      "l_upper_suv_mean","l_upper_suv_sd",
                      "l_lower_suv_mean","l_lower_suv_sd", 
                      "r_upper_suv_max", "l_upper_suv_max",
                      "r_lower_suv_max", "l_lower_suv_max",
                      "side_r0_l1","pneumonitis","pet_distance_ici",
                      "sex_male","size","weight","bmi","pet_impossible_spheres_placement")]

df_filtered_distance <- subset(df_filtered, pet_distance_ici <= 365 | is.na(pet_distance_ici))
df_filtered_distance <- subset(df_filtered_distance, pet_distance_ici >= -1)
df_filtered <- subset(df_filtered_distance, pet_impossible_spheres_placement == 0)
df_filtered[] <- lapply(df_filtered, as.numeric)
#Description: This section filters the data to include only records where distance between PET/CT and immunotherapy is between -1 and 365 days.


#4. Add Columns for Calculations
cols <- c("SUV95","sul_lung_max","sul_lung_mean","LBM","SULmean","SULmax")
df_filtered[,cols] <- NA
# Description: This section adds new columns to the filtered data frame for various calculated metrics.


#5. Define Functions for Calculations
# Define the function to calculate Lean Body Mass (LBM) based on BMI, height, and weight
calculate_LBM <- function(bmi, size, weight, sex) {
  size <- size*100
  if (sex == 1) {
    # male
    lbm <- (9.27 * 10^3 * weight) / ((6.68 * 10^3) + (216 * bmi))
  } else if (sex == 0) {
    # female
    lbm <- (9.27 * 10^3 * weight) / ((8.78 * 10^3) + (244 * bmi))
  } 
  return(lbm)
}

# Define the function to calculate SULmean and SULmax
calculate_SUL <- function(suv, lbm, weight) {
  sul <- NA
  if(!is.na(suv)){
    sul <- (suv / weight) * lbm
  }
  return(sul)
}


#6. Calculate SUV and Related Metrics
for (i in 1:nrow(df_filtered)) {
  if (is.na(df_filtered$l_upper_suv_mean[i]) & !is.na(df_filtered$r_upper_suv_mean[i])) {
    df_filtered$l_upper_suv_mean[i] <- df_filtered$r_upper_suv_mean[i]
    df_filtered$l_upper_suv_sd[i] <- df_filtered$r_upper_suv_sd[i]
  }
  if (is.na(df_filtered$r_upper_suv_mean[i]) & !is.na(df_filtered$l_upper_suv_mean[i])) {
    df_filtered$r_upper_suv_mean[i] <- df_filtered$l_upper_suv_mean[i]
    df_filtered$r_upper_suv_sd[i] <- df_filtered$l_upper_suv_sd[i]
  }
  if (is.na(df_filtered$l_lower_suv_mean[i]) & !is.na(df_filtered$r_lower_suv_mean[i])) {
    df_filtered$l_lower_suv_mean[i] <- df_filtered$r_lower_suv_mean[i]
    df_filtered$l_lower_suv_sd[i] <- df_filtered$r_lower_suv_sd[i]
  }
  if (is.na(df_filtered$r_lower_suv_mean[i]) & !is.na(df_filtered$l_lower_suv_mean[i])) {
    df_filtered$r_lower_suv_mean[i] <- df_filtered$l_lower_suv_mean[i]
    df_filtered$r_lower_suv_sd[i] <- df_filtered$l_lower_suv_sd[i]
  }
}

df_filtered$upper_suv_mean <- rowMeans(df_filtered[, c("r_upper_suv_mean", "l_upper_suv_mean")], na.rm = TRUE)
df_filtered$upper_suv_sd <- rowMeans(df_filtered[, c("r_upper_suv_sd", "l_upper_suv_sd")], na.rm = TRUE)
df_filtered$lower_suv_mean <- rowMeans(df_filtered[, c("r_lower_suv_mean", "l_lower_suv_mean")], na.rm = TRUE)
df_filtered$lower_suv_sd <- rowMeans(df_filtered[, c("r_lower_suv_sd", "l_lower_suv_sd")], na.rm = TRUE)
df_filtered$SUVmean <- rowMeans(df_filtered[, c("upper_suv_mean", "lower_suv_mean")], na.rm = TRUE)
df_filtered$suv_lung_sd <- rowMeans(df_filtered[, c("upper_suv_sd", "lower_suv_sd")], na.rm = TRUE)
df_filtered$upper_suv_max <- rowMeans(df_filtered[, c("r_upper_suv_max", "l_upper_suv_max")], na.rm = TRUE)
df_filtered$lower_suv_max <- rowMeans(df_filtered[, c("r_lower_suv_max", "l_lower_suv_max")], na.rm = TRUE)
df_filtered$SUVmax <- rowMeans(df_filtered[, c("upper_suv_max", "lower_suv_max")], na.rm = TRUE)

for (i in 1:nrow(df_filtered)) {
  df_filtered$SUV95[i] <- df_filtered$SUVmean[i] + (qnorm(0.95) * df_filtered$suv_lung_sd[i])
  df_filtered$LBM[i] <- calculate_LBM(df_filtered$bmi[i], df_filtered$size[i], df_filtered$weight[i], df_filtered$sex_male[i])
  df_filtered$SULmean[i] <- calculate_SUL(df_filtered$SUVmean[i], df_filtered$LBM[i], df_filtered$size[i])
  df_filtered$SULmax[i] <- calculate_SUL(df_filtered$SUVmax[i], df_filtered$LBM[i], df_filtered$size[i])
}
# Description: This section fills in missing SUV values and calculates various metrics based on available data


#7. Generate Plots and Statistical Analysis
# Boxplot for SUVmean
df_filtered$pneumonitis <- ifelse(df_filtered$pneumonitis == 1, "Pneumonitis", "Control")

# Boxplot for SUVmean
t_test <- t.test(SUVmean ~ pneumonitis, data = df_filtered)
p_value <- t_test$p.value
if (round(p_value, 4) > 0.1) {
  p_value_text <- "n. s."
} else {
  p_value_text <- paste("p =", round(p_value, 4))
}

df_filtered_suv <- df_filtered[is.finite(df_filtered$SUVmean), ]
plot_suvmean <- ggplot(data = df_filtered_suv, aes(x = factor(pneumonitis), y = SUVmean)) +
  geom_jitter(width = 0.1, height = 0, alpha = 1, size = .2) +
  stat_summary(fun = mean, geom = "crossbar", width = 0.6, color = "black", linewidth = .2) +
  labs(x = "", y = "SUVmean") +
  annotate("text", x = 1.3, y = 1.6, label = p_value_text,
           hjust = 0, vjust = 1, color = "black") +
  annotate("segment", x = 0.5, xend = 2.5, y = 1.5, yend = 1.5, 
           color = "black", linewidth = 0.5) +
  annotate("segment", x = 0.5, xend = 0.5, y = 1.5, yend =  1.47, 
           color = "black", linewidth = 0.5) +  # Klammer an der linken Seite
  annotate("segment", x = 2.5, xend = 2.5, y = 1.5, yend =  1.47, 
           color = "black", linewidth = 0.5) 

# Boxplot for SUVmax
t_test <- t.test(SUVmax ~ pneumonitis, data = df_filtered)
p_value <- t_test$p.value

if (round(p_value, 4) > 0.1) {
  p_value_text <- "n. s."
} else {
  p_value_text <- paste("p =", round(p_value, 4))
}

df_filtered_suv <- df_filtered[is.finite(df_filtered$SUVmax), ]
plot_suvmax <- ggplot(data = df_filtered_suv, aes(x = factor(pneumonitis), y = SUVmax)) +
  geom_jitter(width = 0.1, height = 0, alpha = 1, size = .2) +
  stat_summary(fun = mean, geom = "crossbar", width = 0.6, color = "black", linewidth = .2) +
  labs(x = "", y = "SUVmax") +
  annotate("text", x = 1.3, y = 3, label = p_value_text,
           hjust = 0, vjust = 1, color = "black") +
  annotate("segment", x = .5, xend = 2.5, y = 2.81, yend = 2.81, 
               color = "black", linewidth = .56) +
  annotate("segment", x = .5, xend = .5, y = 2.81, yend =  2.76, 
               color = "black", linewidth = 0.56) +  
  annotate("segment", x = 2.5, xend = 2.5, y = 2.81, yend =  2.76, 
               color = "black", linewidth = 0.56) 

plot_suvmax

# Boxplot for SUV95
  t_test <- t.test(SUV95 ~ pneumonitis, data = df_filtered)
  p_value <- t_test$p.value
  if (round(p_value, 4) > 0.1) {
    p_value_text <- "n. s."
  } else {
    p_value_text <- paste("p =", round(p_value, 4))
  }
  
  df_filtered_suv <- df_filtered[is.finite(df_filtered$SUV95), ]
  plot_suv95 <- ggplot(data = df_filtered_suv, aes(x = factor(pneumonitis), y = SUV95)) +
    geom_jitter(width = 0.1, height = 0, alpha = 1, size = .2) +
    stat_summary(fun = mean, geom = "crossbar", width = 0.6, color = "black", linewidth = .2) +
    labs(x = "", y = "SUV95") +
  annotate("text", x = 1.3, y = 2.2, label = p_value_text,
            hjust = 0, vjust = 1, color = "black") +
  annotate("segment", x = .5, xend = 2.5, y = 2.055, yend = 2.055, 
               color = "black", linewidth = .56) +
  annotate("segment", x = .5, xend = .5, y = 2.055, yend =  2.02, 
               color = "black", linewidth = .56) +  
  annotate("segment", x = 2.5, xend = 2.5, y = 2.06, yend =  2.02, 
               color = "black", linewidth = 0.56) 
  
# Boxplot for SULMax
t_test <- t.test(SULmax ~ pneumonitis, data = df_filtered)
p_value <- t_test$p.value
if (round(p_value, 4) > 0.1) {
    p_value_text <- "n. s."
} else {
    p_value_text <- paste("p =", round(p_value, 4))
}
  
df_filtered_suv <- df_filtered[is.finite(df_filtered$SULmax), ]
plot_sulmax <- ggplot(data = df_filtered_suv, aes(x = factor(pneumonitis), y = SULmax)) +
  geom_jitter(width = 0.1, height = 0, alpha = 1, size = .2) +
  stat_summary(fun = mean, geom = "crossbar", width = 0.6, color = "black", linewidth = .2) +
  labs(x = "", y = "SULmax") +
  annotate("text", x = 1.3, y = 121, label = p_value_text,
              hjust = 0, vjust = 1, color = "black") +
  annotate("segment", x = .5, xend = 2.5, y = 110, yend = 110, 
                 color = "black", linewidth = .56) +
  annotate("segment", x = .5, xend = .5, y = 110, yend =  108, 
                 color = "black", linewidth = .56) +  
  annotate("segment", x = 2.5, xend = 2.5, y = 110, yend =  108, 
                 color = "black", linewidth = 0.56) 

# Boxplot for SULMean
t_test <- t.test(SULmean ~ pneumonitis, data = df_filtered)
p_value <- t_test$p.value
if (round(p_value, 4) > 0.1) {
  p_value_text <- "n. s."
} else {
  p_value_text <- paste("p =", round(p_value, 4))
}

df_filtered_suv <- df_filtered[is.finite(df_filtered$SULmean), ]
plot_sulmean <- ggplot(data = df_filtered_suv, aes(x = factor(pneumonitis), y = SULmean)) +
  geom_jitter(width = 0.1, height = 0, alpha = 1, size = .2) +
  stat_summary(fun = mean, geom = "crossbar", width = 0.6, color = "black", linewidth = .2) +
  labs(x = "", y = "SULmean") +
  annotate("text", x = 1.3, y = 55, label = p_value_text,
            hjust = 0, vjust = 1, color = "black") +
  annotate("segment", x = .5, xend = 2.5, y = 50, yend = 50, 
               color = "black", linewidth = .56) +
  annotate("segment", x = .5, xend = .5, y = 50, yend =  49, 
               color = "black", linewidth = .56) +  
  annotate("segment", x = 2.5, xend = 2.5, y = 50, yend =  49, 
               color = "black", linewidth = 0.56) 

# Create Grid from Boxplots
combined_plots <- grid.arrange(plot_suvmean, plot_suvmax, plot_suv95, plot_sulmean, plot_sulmax, ncol = 3)
#Description: This section performs statistical tests and generates boxplots for various metrics (SUVmean, SUVmax, SUV95, SULmax, SULmean) and arranges them into a grid for saving.


#8. Save Grid
ggsave("Figure S1.tiff", combined_plots, width = 6, height = 6)
print("saved: Figure S1")
#Saves the grid to a PNG file. 


